/*
 i-net software provides programming examples for illustration only, without warranty
 either expressed or implied, including, but not limited to, the implied warranties
 of merchantability and/or fitness for a particular purpose. This programming example
 assumes that you are familiar with the programming language being demonstrated and
 the tools used to create and debug procedures. i-net software support professionals
 can help explain the functionality of a particular procedure, but they will not modify
 these examples to provide added functionality or construct procedures to meet your
 specific needs.
  
  i-net software 1998-2013

*/
namespace Inet.Viewer
{

    /// <summary>
    /// A single group tree node - representing a group. Stores the level, request type (individual
    /// or all nodes), number of group, request URL (for drilling down), name of group, and
    /// exact position in the report.
    /// </summary>
    public class GroupTreeNode
    {
        /// <summary>
        /// Standard expandable node </summary>
        public const int TypeStandard = 2;
        /// <summary>
        /// Non expandable node, click on this node release drilldown over this group </summary>
        public const int TypeMagnifier = 3;

        /// <summary>
        /// Hierarchical level of node in group tree </summary>
        private byte level;

        /// <summary>
        /// Type of node standard or magnifier </summary>
        private byte requestType;

        /// <summary>
        /// Number of node in group tree </summary>
        private int groupNumber;

        /// <summary>
        /// URL of current node to create a drilldown report </summary>
        private string requestURL;

        /// <summary>
        /// Displayed name of node </summary>
        private string groupName;

        /// <summary>
        /// yFrom y position in a current page, begin of GroupHeader </summary>
        private float yFrom;

        /// <summary>
        /// y position in a current page, end of GroupFooter </summary>
        private float yTo;

        /// <summary>
        /// Page number of begin of GroupHeader </summary>
        private int pageFrom;

        /// <summary>
        /// Page number of end of GroupFooter </summary>
        private int pageTo;

        /// <summary>
        /// Create an empty Group Tree Node (root node).
        /// </summary>
        public GroupTreeNode()
        {
        }

        /// <summary>
        /// Sets this group tree node's group name, level in the tree (0-based), type, group number and request URL for drill-down. </summary>
        /// <param name="level"> hierarchical level of node in group tree </param>
        /// <param name="requestType"> type of current node (standard or magnifier) </param>
        /// <param name="groupNumber"> number of node in group tree </param>
        /// <param name="requestURL"> url of current node to create a drilldown report </param>
        /// <param name="groupName"> displayed name of node </param>
        public virtual void SetName(byte level, byte requestType, int groupNumber, string requestURL, string groupName)
        {
            this.level = level;
            this.requestType = requestType;
            this.groupNumber = groupNumber;
            this.requestURL = requestURL;
            this.groupName = groupName;
        }

        /// <summary>
        /// Sets the position of the group this group tree node represents - a highlight of this group should highlight from position
        /// yFrom on pageFrom to yTo on pageTo. </summary>
        /// <param name="yFrom"> y position in a current page, begin of GroupHeader </param>
        /// <param name="yTo"> y position in a current page, end of GroupFooter </param>
        /// <param name="pageFrom"> page number of begin of GroupHeader </param>
        /// <param name="pageTo"> page number of end of GroupFooter </param>
        public virtual void SetPosition(float yFrom, float yTo, int pageFrom, int pageTo)
        {
            this.yFrom = yFrom;
            this.yTo = yTo;
            this.pageFrom = pageFrom;
            this.pageTo = pageTo;
        }

        /// <returns> Level of this group tree node (0-based) </returns>
        public virtual byte Level
        {
            get
            {
                return level;
            }
        }

        /// <summary>
        /// Type of request for this group tree node - GROUP_TREE_TYPE_INDIVIDUAL_NODE or
        /// GROUPTREE_TYPE_ALL_NODES </summary>
        /// <returns> type of node </returns>
        /// <seealso cref= "TypeMagnifier"/>
        /// <seealso cref= "TypeStandard"/>
        public virtual byte RequestType
        {
            get
            {
                return requestType;
            }
        }

        /// <returns> Number of the group this group tree node represents </returns>
        public virtual int GroupNumber
        {
            get
            {
                return groupNumber;
            }
        }

        /// <returns> Request URL of this group tree node if it is to be drilled down into. </returns>
        public virtual string RequestURL
        {
            get
            {
                return requestURL;
            }
        }

        /// <returns> Name of the group represented by this group tree node </returns>
        public virtual string GroupName
        {
            get
            {
                return groupName;
            }
        }

        /// <summary>
        /// returns the  groupName </summary>  
        /// <returns> node name </returns>
        public override string ToString()
        {
            return groupName;
        }

        /// <returns> Y position in twips of the beginning of the group on the first page this
        /// group starts on. </returns>
        public virtual float YFrom
        {
            get
            {
                return yFrom;
            }
        }

        /// <returns> Y position in twips of the end of the group on the page this group ends on. </returns>
        public virtual float YTo
        {
            get
            {
                return yTo;
            }
        }

        /// <returns> Page this group starts on (1-based) </returns>
        public virtual int PageFrom
        {
            get
            {
                return pageFrom;
            }
        }

        /// <returns> Page this group ends on (1-based) </returns>
        public virtual int PageTo
        {
            get
            {
                return pageTo;
            }
        }
    }
}